﻿using JetBrains.Annotations;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Numerics;
using System.Security.Cryptography;
using System.Text;
using System.Threading.Tasks;
using TMPro;
using UnityEngine;
using UnityEngine.PlayerLoop;
using UnityEngine.Tilemaps;

//CollectionMission represents and manages a collection mission  - collect items around the city
public class CollectionMission : Mission
{
    //CollectionDifficultyData is a container struct for difficulty specific parameters
    public struct CollectionDifficultyData
    {
        public int Time { get; private set; }
        public int Money { get; private set; }
        public List<Vector3Int> ItemPositions { get; private set; }

        public CollectionDifficultyData(int time, int money, List<Vector3Int> itemPositions)
        {
            Time = time;
            Money = money;
            ItemPositions = itemPositions;
        }
    }

    public Dictionary<Constants.MissionDifficulties, CollectionDifficultyData> DifficultyData { get; private set; } = new Dictionary<Constants.MissionDifficulties, CollectionDifficultyData>();

    private float _TimeRemaining = 0;
    private int _ItemsCollected = 0;
    private int _TotalItems = 0;
    private List<Vector3Int> _RemainingItemPositions = new List<Vector3Int>();

    //UI Elements
    private GameObject _BaseUIObject;
    private MissionTimerController _MissionTimer;
    private TMP_Text _ItemsText;

    public CollectionMission(Vector3Int missionSpawnPos) : base(missionSpawnPos)
    {

    }

    public override bool Generate()
    {
        try
        {
            //Let's get the possible destinations
            List<Vector3Int> possibleDestinations = GameController.Instance.LSystem.Drawer.RoadPositions;
            possibleDestinations.Shuffle();

            //And get the data for difficulties
            DifficultyData[Constants.MissionDifficulties.Easy] = GenerateDataForDifficulty(ConfigurationManager.Instance.Missions.Collection.Easy, ConfigurationManager.Instance.Missions.Core.Time.Easy, possibleDestinations);
            DifficultyData[Constants.MissionDifficulties.Medium] = GenerateDataForDifficulty(ConfigurationManager.Instance.Missions.Collection.Medium, ConfigurationManager.Instance.Missions.Core.Time.Medium, possibleDestinations);
            DifficultyData[Constants.MissionDifficulties.Hard] = GenerateDataForDifficulty(ConfigurationManager.Instance.Missions.Collection.Hard, ConfigurationManager.Instance.Missions.Core.Time.Hard, possibleDestinations);
            IsGenerated = true;
            return true;
        }

        catch (Exception ex)
        {
            Debug.LogError("ERROR: Caught an exception when generating collection mission, returning false. The exception is: " + ex);
            return false;
        }
    }

    public override void StartMission(Constants.MissionDifficulties difficulty)
    {
        //Store the difficulty, update the minimap
        SelectedDifficulty = difficulty;
        foreach(Vector3Int destination in DifficultyData[SelectedDifficulty].ItemPositions)
        {
            MinimapManager.Instance.AddIcon(destination, Constants.CollectionItemDestinationPrefix + destination.ToString(), ColoursManager.Instance.Colours["CollectionItemIcon"].Colour, ConfigurationManager.Instance.Missions.Collection.DestinationIconRenderSize, 0, Resources.Load<Sprite>("Minimap/Icons/minimapDestinationIcon"));
        }
        
        _TimeRemaining = DifficultyData[SelectedDifficulty].Time;
        _RemainingItemPositions = DifficultyData[SelectedDifficulty].ItemPositions;
        _TotalItems = _RemainingItemPositions.Count;

        //Setup the UI
        _BaseUIObject = GameManager.Instance.MissionsUIGameObject.FindChild("Collection");
        _MissionTimer = _BaseUIObject.FindChild("HeaderFlyout").FindChild("Timer").GetComponent<MissionTimerController>();
        _ItemsText = _BaseUIObject.FindChild("ItemsFlyout").FindChild("ItemsText").GetComponent<TMP_Text>();

        _ItemsText.text = "ITEMS: " + _ItemsCollected + " / " + _TotalItems;
        _ItemsText.gameObject.SetActive(true);

        //Add warning to timer from 10s downwards
        for (int i = 10000; i >= 0; i -= 1000)
        {
            _MissionTimer.AddTime(i);
        }

        //Start the timer and the mission
        _MissionTimer.StartTimer(DifficultyData[SelectedDifficulty].Time);

        UpdateTimeUI();
        _BaseUIObject.SetActive(true);
        _IsMissionStarted = true;
    }

    public override void EndMission(bool passed)
    {
        //Let's tidy up the minimap
        foreach (Vector3Int pos in DifficultyData[SelectedDifficulty].ItemPositions.ToList())
        {
            MinimapManager.Instance.RemoveIcon(Constants.CollectionItemDestinationPrefix + pos.ToString());
        }

        //And reward the player if they passed
        if(passed)
        {
            GameManager.Instance.IncrementMoney(DifficultyData[SelectedDifficulty].Money);
        }

        //Tidy up the rest of the components and stop
        _MissionTimer.StopTimer();
        _RemainingItemPositions.Clear();
        _ItemsCollected = 0;
        _BaseUIObject.SetActive(false);
        _IsMissionStarted = false;
    }

    public override string GetRewardText()
    {
        if(SelectedDifficulty != Constants.MissionDifficulties.Sentinel)
        {
            return "+$" + DifficultyData[SelectedDifficulty].Money;
        }

        return "";
    }

    public override void SpawnMinimapIcon(string iconID)
    {
        MinimapManager.Instance.AddIcon(MissionSpawnPosition, iconID, ColoursManager.Instance.Colours["CollectionMissionIcon"].Colour, ConfigurationManager.Instance.Minimap.MissionIconRenderSize, 0, Resources.Load<Sprite>("Minimap/Icons/minimapCollectionIcon"));
    }

    public override void Update()
    {
        if (_IsMissionStarted)
        {
            //Decrement the time left, update the UI
            _TimeRemaining -= (Time.deltaTime * 1000.0f);
            UpdateTimeUI();

            if (_TimeRemaining <= 0.0f)
            {
                //Out of time, mission failed
                MissionsManager.Instance.EndMission(false);
            }

            else
            {
                Vector3Int tilePos = GameManager.Instance.RoadsTilemap.WorldToCell(GameManager.Instance.PlayerCarGameObject.transform.position);

                //Loop through all our item positions
                foreach(Vector3Int pos in _RemainingItemPositions.ToList())
                {
                    if(tilePos == pos)
                    {
                        //The player is over an item, let's pick it up, remove it and the icon and play the SFX
                        _RemainingItemPositions.Remove(pos);
                        _ItemsCollected++;
                        _ItemsText.text = "ITEMS: " + _ItemsCollected + " / " + _TotalItems;
                        MinimapManager.Instance.RemoveIcon(Constants.CollectionItemDestinationPrefix + pos.ToString());
                        AudioManager.Instance.PlayFile(AudioManager.Instance.LoadedFiles["CollectedSFX"]);

                        if(_ItemsCollected == _TotalItems)
                        {
                            //We've collected all items, mission passed!
                            MissionsManager.Instance.EndMission(true);
                        }
                    }
                }
            }
        }
    }

    private CollectionDifficultyData GenerateDataForDifficulty(CollectionDifficultyConfiguration missionConfig, MissionsTimeDifficultyConfiguration timeConfig, List<Vector3Int> possibleDestinations)
    {
        //Generate a reward amount of money
        Randomizer.Regenerate();
        int generatedMoney = Convert.ToInt32(Math.Round(Randomizer.RNG.Next(missionConfig.MoneyLowerRange, missionConfig.MoneyUpperRange) / (float)ConfigurationManager.Instance.Missions.Core.MoneyRoundFactor) * ConfigurationManager.Instance.Missions.Core.MoneyRoundFactor);

        //Generate a random number of destinations
        Randomizer.Regenerate();
        int generatedDestinationsCount = Randomizer.RNG.Next(missionConfig.DestinationsLowerRange, missionConfig.DestinationsUpperRange);

        //Generate the destinations themselves
        List<Vector3Int> generatedDestinations = new List<Vector3Int>();

        for(int i = 0; i < generatedDestinationsCount; i++)
        {
            Randomizer.Regenerate();
            int generatedIndex = Randomizer.RNG.Next(0, possibleDestinations.Count);
            generatedDestinations.Add(possibleDestinations[generatedIndex]);
        }

        //Loop through all of our positions to find the greatest distance between destinations
        float largestDistance = 0.0f;
        Vector3Int firstPosition = Constants.SentinelVector3Int;
        Vector3Int secondPosition = Constants.SentinelVector3Int;

        foreach (Vector3Int firstDistance in generatedDestinations)
        {
            foreach(Vector3Int secondDistance in generatedDestinations)
            {
                float thisDistance = Vector3Int.Distance(firstDistance, secondDistance);

                if (thisDistance > largestDistance)
                {
                    firstPosition = firstDistance;
                    secondPosition = secondDistance;
                    largestDistance = thisDistance;
                }
            }
        }

        //Found the largest destination, use it and the number of destinations for time calculations
        int generatedTime = GenerateTimeForDifficulty(timeConfig, firstPosition, secondPosition) + (missionConfig.TimePerDestination * generatedDestinationsCount);
        generatedTime = Convert.ToInt32(Math.Ceiling(generatedTime * (1.0f / ConfigurationManager.Instance.Missions.Core.Time.RoundFactor)) * ConfigurationManager.Instance.Missions.Core.Time.RoundFactor);

        return new CollectionDifficultyData(generatedTime, generatedMoney, generatedDestinations);
    }

    private void UpdateTimeUI()
    {
        _MissionTimer.UpdateTimer(_TimeRemaining);
    }
}
